/**
 * Translator object that handles translation of individual texts.
 * In order to work, the global variable 'translations' must be defined.
 * The optional global variable 'page_translations' may contain page-specific
 * translations that override those in 'translations'.
 * Both global variables are deleted once object is constructed. 
 */
Ext.define('CCS.Translator', {
	singleton: true,
	translations: {  // default translations if needed for technical reasons
	},
	commontranslations: {
	},
	
	/**
	 * Character mark to put before untranslated texts, default blank.
	 */
	mark: '',
	
	constructor: function(config) {
		if (typeof(translations) == 'object') {
			Ext.Object.merge(this.translations, translations);
			delete translations;
		} 

		if (typeof(commontranslations) == 'object') {
			//console.log(commontranslations);
			Ext.Object.merge(this.translations, commontranslations);
			delete commontranslations;
		} 
	},
	
	/**
	 * @private
	 * Replaces a variable number of placeholders in text with values from replacementValues array.
	 * Placeholders must be formatted {1}, {2}, {3}, etc. and numbering begins with 1! 
	 *
	 * @param {String} text string with placeholders
	 * @param {Array} [replacementValues] array of values to replace placeholders
	 * @return {String} 
	 */
	replacePlaceholders: function(text, replacementValues) {
		// replaces a variable number of placeholders in text with values from replacementValues array
		// placeholders must be formatted {1}, {2}, {3}, etc. - numbers start from 1!
		for (i = 0; i < replacementValues.length; i++) {
			// replace {i} with arguments[i]
			text = text.replace('{' + (i+1) + '}', replacementValues[i]);
		}
		return text;
	},
	
	/**
	 * @public
	 * Translates the given English text into the target language, and optionally replaces
	 * placeholders with the given replacement values.
	 *
	 * Placeholders must be formatted {1}, {2}, {3}, etc. 
	 * Note that placeholder numbers start from 1!
	 *
	 * @param {String} english language text to be translated
	 * @param {Array} [replacementValues] array of values to replace placeholders
	 * @return {String} 
	 */
	getText: function(english, replacementValues) {
		var target = english;
		
		if (typeof(english) === 'string' && english.length > 0 && english.indexOf("<img src=") < 0) {
			target = this.translations[english];
			if ((target === undefined || target == '') && this.mark.length > 0) {
				target = this.isNotAlreadyTranslated(english)? this.mark + english : english;
			}
			else if (target === undefined || target == '')
				target = english;
		}

		// perform replacements if applicable
		if (replacementValues !== undefined) {
			target = this.replacePlaceholders(target, replacementValues);
		}
		
		return target;
	},
	
	isNotAlreadyTranslated: function(english) {
		var notInTranslation = true;
		
		for(var x in this.translations) {
			if(this.translations[x] == english) //string has already been translated
				notInTranslation = false;
		}
		
		return notInTranslation;
	},
	
	
	/**
	 * @public
	 * Translates the given property of the given object. This is mainly intended for use
	 * in auto-translation overrides. The object property is changed in-place.
	 * 
	 * @param {Object} object
	 * @param {String} property name of property to translate
	 */
	translate: function(object, property) {
		// translates a property value of the given object
		if (object && typeof(object[property]) == 'string') {
			object[property] = this.getText(object[property]);
		}
	}
});

/*
 * UI widget-specific overrides that call CCS.Translator to perform translations. 
 */

Ext.override(Ext.panel.Panel, {
	constructor: function(config) {
		// sometimes called with config.title, other times object is already configured with title property
		if (config && config.title) 
			CCS.Translator.translate(config, 'title');
		else 
			CCS.Translator.translate(this, 'title');
		return this.callParent(arguments);
	},
	setTitle: function(title) {
		title = CCS.Translator.getText(title);
		return this.callParent(arguments);		
	}
});

Ext.override(Ext.panel.Tool, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'tooltip');
		return this.callParent(arguments);
	}
});

/* Window override not necessary - handled by Panel (superclass)
Ext.override(Ext.window.Window, {
	constructor: function(config) {
		if (config) console.log("Window constructor: " + config.title); else console.log("Window constructor");
		CCS.Translator.translate(config, 'title');
		return this.callParent(arguments);
	},
	setTitle: function(title) {
		console.log("Window setTitle: " + title);
		title = CCS.Translator.getText(title);
		return this.callParent(arguments);		
	}
});
*/

/* Tab override not necessary - handled by Button
Ext.override(Ext.tab.Tab, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'text');
		return this.callParent(arguments);
	},
	setText: function(text) {
		text = CCS.Translator.getText(text);
		return this.callParent(arguments);		
	}
});
*/

/*Ext.override(Ext.form.field.Radio, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'boxLabel');
		return this.callParent(arguments);
	}
});*/

Ext.override(Ext.button.Button, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'text');
		CCS.Translator.translate(config, 'tooltip');
		return this.callParent(arguments);
	}
/* setText override not necessary and leads to double-translation of tabs	
	setText: function(text) {
		console.log("Button setText: " + text);
		text = CCS.Translator.getText(text);
		return this.callParent(arguments);		
	}
*/
});

Ext.override(Ext.form.FieldSet, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'title');
		return this.callParent(arguments);
	},
	setTitle: function(title) {
		title = CCS.Translator.getText(title);
		return this.callParent(arguments);		
	}
});

Ext.override(Ext.grid.column.Column, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'text');
		CCS.Translator.translate(config, 'tooltip');
		return this.callParent(arguments);
	}
});

Ext.override(Ext.form.field.Base, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'fieldLabel');
		CCS.Translator.translate(config, 'boxLabel');
		return this.callParent(arguments);
	}
});

Ext.override(Ext.form.field.Text, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'emptyText');
		return this.callParent(arguments);
	}
});

Ext.override(Ext.form.FieldContainer, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'fieldLabel');
		return this.callParent(arguments);
	}
});

Ext.override(Ext.form.CheckboxGroup, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'fieldLabel');
		return this.callParent(arguments);
	}
});

Ext.override(Ext.form.Label, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'text');
		return this.callParent(arguments);
	}
});

//TODO quick tips - cannot seem to get a handle on these => need to update these in own code instead

Ext.override(Ext.chart.axis.Axis, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'title');
		return this.callParent(arguments);
	}
});

Ext.override(Ext.chart.series.Series, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'title');
		return this.callParent(arguments);
	}
});

Ext.override(Ext.window.MessageBox, {
	alert: function(title, msg) {
		msg = CCS.Translator.getText(msg);
		return this.callParent(arguments);
	}
});

Ext.override(Ext.Img, {
	constructor: function(config) {
		CCS.Translator.translate(config, 'title');
		return this.callParent(arguments);
	}
});